<?php
if ( ! defined( 'ABSPATH' ) ) {
	exit; // Exit if accessed directly.
}

/**
 * Main Piller Core Class
 *
 * The main class that initiates and runs the plugin.
 *
 * @since 1.0.0
 */

final class Piller_Extension {

	/**
	 * Plugin Version
	 *
	 * @since 1.0.0
	 *
	 * @var string The plugin version.
	 */
	const VERSION = '1.0.0';

	/**
	 * Minimum Elementor Version
	 *
	 * @since 1.0.0
	 *
	 * @var string Minimum Elementor version required to run the plugin.
	 */

	const MINIMUM_ELEMENTOR_VERSION = '2.0.0';

	/**
	 * Minimum PHP Version
	 *
	 * @since 1.0.0
	 *
	 * @var string Minimum PHP version required to run the plugin.
	 */
	const MINIMUM_PHP_VERSION = '7.0';


	/**
	 * Instance
	 *
	 * @since 1.0.0
	 *
	 * @access private
	 * @static
	 *
	 * @var Elementor_Test_Extension The single instance of the class.
	 */

	private static $_instance = null;

	/**
	 * Instance
	 *
	 * Ensures only one instance of the class is loaded or can be loaded.
	 *
	 * @since 1.0.0
	 *
	 * @access public
	 * @static
	 *
	 * @return Elementor_Test_Extension An instance of the class.
	 */
	public static function instance() {

		if ( is_null( self::$_instance ) ) {
			self::$_instance = new self();
		}
		return self::$_instance;

	}

	/**
	 * Constructor
	 *
	 * @since 1.0.0
	 *
	 * @access public
	 */
	public function __construct() {
		add_action( 'plugins_loaded', [ $this, 'init' ] );
	}

	/**
	 * Initialize the plugin
	 *
	 * Load the plugin only after Elementor (and other plugins) are loaded.
	 * Checks for basic plugin requirements, if one check fail don't continue,
	 * if all check have passed load the files required to run the plugin.
	 *
	 * Fired by `plugins_loaded` action hook.
	 *
	 * @since 1.0.0
	 *
	 * @access public
	 */
	public function init() {

		// Check if Elementor installed and activated

		if ( ! did_action( 'elementor/loaded' ) ) {
			add_action( 'admin_notices', [ $this, 'admin_notice_missing_main_plugin' ] );
			return;
		}

		// Check for required Elementor version

		if ( ! version_compare( ELEMENTOR_VERSION, self::MINIMUM_ELEMENTOR_VERSION, '>=' ) ) {
			add_action( 'admin_notices', [ $this, 'admin_notice_minimum_elementor_version' ] );
			return;
		}

		// Check for required PHP version

		if ( version_compare( PHP_VERSION, self::MINIMUM_PHP_VERSION, '<' ) ) {
			add_action( 'admin_notices', [ $this, 'admin_notice_minimum_php_version' ] );
			return;
		}


		// Add Plugin actions

		add_action( 'elementor/widgets/register', [ $this, 'init_widgets' ] );


        // Register widget scripts

		add_action( 'elementor/frontend/after_enqueue_scripts', [ $this, 'widget_scripts' ]);


		// Specific Register widget scripts

		// add_action( 'elementor/frontend/after_register_scripts', [ $this, 'piller_regsiter_widget_scripts' ] );
		// add_action( 'elementor/frontend/before_register_scripts', [ $this, 'piller_regsiter_widget_scripts' ] );


        // category register

		add_action( 'elementor/elements/categories_registered',[ $this, 'piller_elementor_widget_categories' ] );
	}

	/**
	 * Admin notice
	 *
	 * Warning when the site doesn't have Elementor installed or activated.
	 *
	 * @since 1.0.0
	 *
	 * @access public
	 */
	public function admin_notice_missing_main_plugin() {

		if ( isset( $_GET['activate'] ) ) unset( $_GET['activate'] );

		$message = sprintf(
			/* translators: 1: Plugin name 2: Elementor */
			esc_html__( '"%1$s" requires "%2$s" to be installed and activated.', 'piller' ),
			'<strong>' . esc_html__( 'Piller Core', 'piller' ) . '</strong>',
			'<strong>' . esc_html__( 'Elementor', 'piller' ) . '</strong>'
		);

		printf( '<div class="notice notice-warning is-dismissible"><p>%1$s</p></div>', $message );
	}

	/**
	 * Admin notice
	 *
	 * Warning when the site doesn't have a minimum required Elementor version.
	 *
	 * @since 1.0.0
	 *
	 * @access public
	 */
	public function admin_notice_minimum_elementor_version() {

		if ( isset( $_GET['activate'] ) ) unset( $_GET['activate'] );

		$message = sprintf(
			/* translators: 1: Plugin name 2: Elementor 3: Required Elementor version */

			esc_html__( '"%1$s" requires "%2$s" version %3$s or greater.', 'piller' ),
			'<strong>' . esc_html__( 'Piller Core', 'piller' ) . '</strong>',
			'<strong>' . esc_html__( 'Elementor', 'piller' ) . '</strong>',
			 self::MINIMUM_ELEMENTOR_VERSION
		);

		printf( '<div class="notice notice-warning is-dismissible"><p>%1$s</p></div>', $message );
	}
	/**
	 * Admin notice
	 *
	 * Warning when the site doesn't have a minimum required PHP version.
	 *
	 * @since 1.0.0
	 *
	 * @access public
	 */
	public function admin_notice_minimum_php_version() {

		if ( isset( $_GET['activate'] ) ) unset( $_GET['activate'] );

		$message = sprintf(

			/* translators: 1: Plugin name 2: PHP 3: Required PHP version */
			esc_html__( '"%1$s" requires "%2$s" version %3$s or greater.', 'piller' ),
			'<strong>' . esc_html__( 'Piller Core', 'piller' ) . '</strong>',
			'<strong>' . esc_html__( 'PHP', 'piller' ) . '</strong>',
			 self::MINIMUM_PHP_VERSION
		);

		printf( '<div class="notice notice-warning is-dismissible"><p>%1$s</p></div>', $message );
	}

	/**
	 * Init Widgets
	 *
	 * Include widgets files and register them
	 *
	 * @since 1.0.0
	 *
	 * @access public
	 */

	public function init_widgets() {

		$widget_register = \Elementor\Plugin::instance()->widgets_manager;

		// Header Include file & Widget Register
		require_once( PILLER_ADDONS . '/header/header.php' );
		require_once( PILLER_ADDONS . '/header/header2.php' );

		$widget_register->register ( new \Piller_Header() );
		$widget_register->register ( new \Piller_Header2() );


		// Include All Widget Files
		foreach($this->Piller_Include_File() as $widget_file_name){
			require_once( PILLER_ADDONS . '/widgets/piller-'."$widget_file_name".'.php' );
		}
		// All Widget Register
		foreach($this->Piller_Register_File() as $name){
			$widget_register->register ( $name );
		}
		
	}

	public function Piller_Include_File(){
		return [
			'banner', 
			'banner2', 
			'section-title', 
			'button', 
			'blog', 
			'blog-page', 
			'service', 
			'testimonial', 
			'team', 
			'team-info', 
			'image', 
			'contact-info', 
			'contact-form', 
			'counterup', 
			'faq', 
			'client-logo', 
			'cta', 
			'gallery', 
			'info-box', 
			'newsletter', 
			'menu-select',
			'footer-widgets',
			'footer-copyright',

			'listing-single', 
			'listing-gallery', 
			'listing-features', 
			'listing-floorplan', 
			'listing-video', 
			'listing-action',
			'listing-search',

			'listing-filter',  
			'listing-grid', 
			'listing-slider', 

			'video', 
			'features', 
			'animated-shape', 
			'choose-us',
			'destination',
			'category',
			'price',
			'marquee',
 
			'social',
			'arrows', 
			'tab-builder', 
			'megamenu',
		];
	}

	public function Piller_Register_File(){
		return [
			new \Piller_Banner1(),
			new \Piller_Banner2(),
			new \Piller_Section_Title(),
			new \Piller_Button(),
			new \Piller_Blog(),
			new \Piller_Blog_Page(),
			new \Piller_Service(),
			new \Piller_Testimonial(),
			new \Piller_Team(),
			new \Piller_Team_info(),
			new \Piller_Image(),
			new \Piller_Contact_Info(),
			new \Piller_Contact_Form(),
			new \Piller_Counterup(),
			new \Piller_Faq(),
			new \Piller_Client_Logos(), 
			new \Piller_Cta(),
			new \Piller_Gallery(),
			new \Piller_Info_Box(),
			new \piller_Newsletter(),
			new \Piller_Menu(),
			new \Piller_Footer_Widgets(),
			new \Piller_Copyright(),

			new \Piller_Listing_Single(),
			new \Piller_Listing_Gallery(),
			new \Piller_Listing_Features(),
			new \Piller_Listing_FloorPlan(),
			new \Piller_Listing_Video(),
			new \Piller_Single_quick_Info(),
			new \Piller_Listing_Search(),

			new \Piller_Listing_Filter(),
			new \Piller_Listing_Grid(),
			new \Piller_Listing_Slider(),

			new \piller_Video(),
			new \Piller_Features(),
			new \Piller_Animated_Shape(),
			new \piller_Choose_Us(),
			new \Piller_Destination(),
			new \Piller_Category(),
			new \Piller_Price(),
			new \Piller_Marquee(),
			
			new \Piller_Social(),
			new \Piller_Arrows(),
			new \Piller_Tab_Builder(),
			new \Piller_Megamenu(),
		];
	}

    public function widget_scripts() {

        // wp_enqueue_script(
        //     'piller-frontend-script',
        //     PILLER_PLUGDIRURI . 'assets/js/piller-frontend.js',
        //     array('jquery'),
        //     false,
        //     true
		// );

	}


    function piller_elementor_widget_categories( $elements_manager ) {

        $elements_manager->add_category(
            'piller',
            [
                'title' => __( 'Piller', 'piller' ),
                'icon' 	=> 'fa fa-plug',
            ]
        );

        $elements_manager->add_category(
            'piller_footer_elements',
            [
                'title' => __( 'Piller Footer Elements', 'piller' ),
                'icon' 	=> 'fa fa-plug',
            ]
		);

		$elements_manager->add_category(
            'piller_header_elements',
            [
                'title' => __( 'Piller Header Elements', 'piller' ),
                'icon' 	=> 'fa fa-plug',
            ]
        );
	}
}

Piller_Extension::instance();